/*
 * Copyright (C) 2000-2025 the xine project
 *
 * This file is part of xine, a unix video player.
 *
 * xine is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 2 of the License, or
 * (at your option) any later version.
 *
 * xine is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 51 Franklin Street, Fifth Floor, Boston, MA 02110, USA
 *
 */
#ifdef HAVE_CONFIG_H
#include "config.h"
#endif

#include <stdio.h>
#include <string.h>

#include "_xitk.h"
#include "combo.h"
#include "button.h"
#include "label.h"
#include "browser.h"
#include "font.h"
#include "skin.h"
#include "_backend.h"

#define _COMBO_BROWSER_LINES 5

typedef enum {
  _W_combo = 0,
  /* keep order */
  _W_label,
  _W_button,
  /* /keep order */
  _W_browser,
  _W_LAST
} _W_t;

typedef struct {
  xitk_widget_t           w;

  xitk_window_t          *xwin;

  int                     win_x;
  int                     win_y;

  xitk_register_key_t    *parent_wkey;

  xitk_widget_t          *iw[_W_LAST];

  char                  **entries;
  int                     num_entries;
  int                     selected;

  uint32_t                last_pos;

  xitk_register_key_t     widget_key;
  xitk_widget_list_t     *widget_list;

  xitk_int_callback_t     callback;
  int                     ignore_click;

} _combo_private_t;

static char **_combo_copy_string_list (const char * const *s, int *n) {
  const char * const *p;
  char **q, **r, *d;
  size_t _n, _s;

  if (!s || (*n <= 0)) {
    *n = 0;
    return NULL;
  }

  for (_n = *n, _s = 0, p = s; *p && (_n > 0); p++, _n--)
    _s += sizeof (*p) + xitk_find_byte (*p, 0) + 1;
  _n = *n - _n;
  *n = 0;

  r = malloc (_s + sizeof (*p));
  if (!r)
    return NULL;

  *n = _n;
  p = s;
  q = r;
  d = (char *)(r + _n + 1);
  while (_n--) {
    size_t _l = xitk_find_byte (*p, 0) + 1;
    *q = d;
    memcpy (d, *p, _l);
    d += _l;
    p++;
    q++;
  }
  *q = NULL;
  return r;
}

static void _combo_close (_combo_private_t *wp, int focus) {
  if (wp->xwin) {
    xitk_widget_register_win_pos (&wp->w, 0);
    if (focus)
      xitk_window_set_input_focus (wp->w.wl->xwin);
    wp->iw[_W_browser] = NULL;
    xitk_unregister_event_handler (wp->w.wl->xitk, &wp->widget_key);
    xitk_window_destroy_window (wp->xwin);
    wp->xwin = NULL;
    if (focus) {
      xitk_set_focus_to_widget (wp->iw[_W_button]);
      xitk_button_set_state (wp->iw[_W_button], 0);
    }
  }
}

/*
 * Handle Xevents here.
 */

static int combo_event (void *data, const xitk_be_event_t *e) {
  _combo_private_t *wp = (_combo_private_t*)data;

  switch (e->type) {
    case XITK_EV_SHOW:
      /* dont annoy the user too much ;-) */
      wp->widget_list->flags &= ~XITK_WL_NO_MOUSE_FOCUS;
      return 0;
    case XITK_EV_DEL_WIN:
    case XITK_EV_UNFOCUS:
      _combo_close (wp, 1);
      if ((wp->w.wl->widget_under_mouse == wp->iw[_W_button])
        || (wp->w.wl->widget_under_mouse == wp->iw[_W_label]))
        wp->ignore_click = 1;
      return 1;
    case XITK_EV_KEY_UP:
      return (e->utf8[0] == XITK_CTRL_KEY_PREFIX) && (e->utf8[1] == XITK_KEY_ESCAPE);
    case XITK_EV_KEY_DOWN:
      if (e->utf8[0] == XITK_CTRL_KEY_PREFIX) {
        switch (e->utf8[1]) {
          case XITK_KEY_ESCAPE:
            _combo_close (wp, 1);
            return 1;
          default:
            return xitk_widget_key_event (wp->iw[_W_browser], e, 0);
        }
      }
      break;
    default: ;
  }
  return 0;
}

static void _combo_select (xitk_widget_t *w, void *data, int selected, unsigned int modifier) {
  _combo_private_t *wp = (_combo_private_t *)data;
#if 0
  /* this is a private callback, and xitk wont misuse. */
  if (!wp || !w)
    return;
  if ((w->type & WIDGET_TYPE_MASK) != WIDGET_TYPE_BROWSER)
    return;
#else
  (void)w;
#endif
  if (modifier & (MODIFIER_SHIFT | MODIFIER_CTRL | MODIFIER_META))
    return;
  if (!wp->xwin)
    return;

  _combo_close (wp, 1);
  if (selected < 0)
    return;
  wp->selected = selected;
  xitk_label_change_label (wp->iw[_W_label], wp->entries[selected]);
  if (wp->callback)
    wp->callback (wp->iw[_W_combo], wp->w.userdata, selected, modifier);
}

static void _combo_open (_combo_private_t *wp) {
  int itemw, itemh, slidw;
  XITK_HV_INIT;

  if (wp->xwin)
    return;

  /* sigh. a click while cmbo window is open means that
   * the latter has lost focus, closed itself, and turned off the button
   * (or will soon with a buggy window manager).
   * instead of turning on and reoopening again here,
   * ignore and reset here. */
  if (wp->ignore_click) {
    wp->ignore_click = 0;
    xitk_button_set_state (wp->iw[_W_button], 0);
    return;
  }

  itemh = XITK_HV_V (wp->iw[_W_label]->size);
  if (itemh < 18)
    itemh = 18;
  slidw = 12;
  itemw = XITK_HV_H (wp->w.size) - 3; /* space for border */

  {
    window_info_t wi;
    int dw, dh, wh;

    xitk_get_display_size (wp->w.wl->xitk, &dw, &dh);
    wp->win_x = XITK_HV_H (wp->iw[_W_label]->pos);
    wp->win_y = XITK_HV_V (wp->iw[_W_label]->pos);
    if ((xitk_get_window_info (wp->w.wl->xitk, *(wp->parent_wkey), &wi)))
      wp->win_x += wi.x, wp->win_y += wi.y;
    wh = XITK_HV_V (wp->iw[_W_label]->size);
    wp->win_y += (wp->win_y + wh + itemh * 5 + 3 <= dh)
               ? wh /* below */
               : -itemh * 5 - 3; /* above */
  }

  wp->xwin = xitk_window_create_window_ext (wp->w.wl->xitk,
    wp->win_x, wp->win_y, itemw + 3, itemh * 5 + 3,
    NULL, "Xitk Combo", "Xitk", 1, 0, NULL, XITK_WINDOW_BG_SIMPLE);
  if (!wp->xwin)
    return;

  xitk_window_flags (wp->xwin, XITK_WINF_FIXED_POS, XITK_WINF_FIXED_POS);

  wp->widget_list        = xitk_window_widget_list(wp->xwin);

  {
    /* Browser */
    char fontname[128];
    xitk_browser_widget_t browser = {
      .nw = {
        .wl       = wp->widget_list,
        .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE,
        .mode_mask = WIDGET_GROUP_MEMBER | WIDGET_GROUP_COMBO,
        .mode_value = WIDGET_GROUP_MEMBER | WIDGET_GROUP_COMBO,
        .userdata = wp
      },
      .browser = {
        .num_entries = wp->num_entries,
        .entries     = (const char * const *)wp->entries,
        .max_displayed_entries = _COMBO_BROWSER_LINES
      },
      .callback = _combo_select
    };
    XITK_DEFAULT_FONT_NAME (fontname, DEFAULT_FONT_FMT, itemh);
    wp->iw[_W_browser] = xitk_noskin_browser_create (&browser,
      1, 1, itemw, itemh, -slidw, fontname);
  }
  xitk_browser_set_select (wp->iw[_W_browser], wp->selected);
  xitk_widgets_state (wp->iw + _W_browser, 1, XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE, ~0u);

  wp->widget_key = xitk_be_register_event_handler ("xitk combo", wp->xwin, combo_event, wp, NULL, NULL);
  xitk_window_set_transient_for_win (wp->xwin, wp->w.wl->xwin);

  xitk_window_flags (wp->xwin, XITK_WINF_VISIBLE | XITK_WINF_ICONIFIED, XITK_WINF_VISIBLE);
  xitk_window_raise_window (wp->xwin);
  xitk_window_set_input_focus (wp->xwin);

  /* No widget focused, give focus to the first one */
  if (wp->widget_list->widget_focused == NULL)
    xitk_set_focus_to_next_widget (wp->widget_list, 0, 0);

  xitk_widget_register_win_pos (&wp->w, 1);
}

/*
 *
 */
static void _combo_rollunroll (xitk_widget_t *w, void *data, int state, unsigned int modifier) {
  _combo_private_t *wp = (_combo_private_t *)data;

  (void)modifier;
#if 0
  if (wp && w && (((w->type & WIDGET_GROUP_MASK) & WIDGET_GROUP_COMBO) &&
    ((w->type & WIDGET_TYPE_MASK) == WIDGET_TYPE_CHECKBOX))) {
    /* no typo, thats what it reports with state ^ */
#else
  (void)w;
#endif
    if (state) {
      _combo_open (wp);
    } else {
      _combo_close (wp, 0);
    }
#if 0
  }
#endif
}

/*
 *
 */
static void _combo_paint (_combo_private_t *wp, uint32_t flags) {
  XITK_HV_INIT;

  if ((wp->w.state & flags) && (wp->w.pos.w != wp->last_pos)) {
    int bx, lw;

    wp->last_pos = wp->w.pos.w;
    lw = XITK_HV_H (wp->iw[_W_label]->size);
    xitk_set_widget_pos (wp->iw[_W_label], XITK_HV_H (wp->w.pos), XITK_HV_V (wp->w.pos));
    bx = XITK_HV_H (wp->w.pos) + lw;
    xitk_set_widget_pos (wp->iw[_W_button], bx, XITK_HV_V (wp->w.pos));
  } else {
    xitk_button_set_state (wp->iw[_W_button], 0);
    _combo_close (wp, 0);
  }
  /* label, button */
  xitk_widgets_state (wp->iw + _W_label, 2, XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE, wp->w.state);
}

/*
 *
 */
static void _combo_hull (_combo_private_t *wp) {
  uint32_t v = wp->w.pos.w = wp->iw[_W_label]->pos.w;
  v += wp->w.size.w = wp->iw[_W_label]->size.w;
  xitk_set_widget_pos (wp->iw[_W_button], v & 0xffff, wp->w.pos.w >> 16);
  wp->w.size.w += wp->iw[_W_button]->size.w & 0xffff;
}

static int notify_event(xitk_widget_t *w, const widget_event_t *event) {
  _combo_private_t *wp;
  uint32_t flags;

  xitk_container (wp, w, w);
  if (!wp || !event)
    return 0;
  if ((wp->w.type & WIDGET_TYPE_MASK) != WIDGET_TYPE_COMBO)
    return 0;

  switch (event->type) {
    case WIDGET_EVENT_ENABLE: /* no joke :-) */
      flags = XITK_WIDGET_STATE_ENABLE;
      goto _event_flags;
    case WIDGET_EVENT_PAINT:
      flags = XITK_WIDGET_STATE_VISIBLE;
    _event_flags:
      _combo_paint (wp, flags);
      break;
    case WIDGET_EVENT_CHANGE_SKIN:
      if (wp->w.skin_element_name[0]) {
        xitk_widget_state_from_info (&wp->w, xitk_skin_get_info (event->skonfig, wp->w.skin_element_name));
        _combo_hull (wp);
      }
      break;
    case WIDGET_EVENT_DESTROY:
      _combo_close (wp, 0);
      xitk_widgets_delete (wp->iw + _W_label, 2);
      XITK_FREE (wp->entries);
      break;
    case WIDGET_EVENT_TIPS_TIMEOUT:
      xitk_set_widget_tips_and_timeout (wp->iw[_W_button], wp->w.tips_string, event->tips_timeout);
      break;
    case WIDGET_EVENT_SELECT:
      if (XITK_0_TO_MAX_MINUS_1 (event->button, wp->num_entries) && (event->button != wp->selected)) {
        wp->selected = event->button;
        xitk_label_change_label (wp->iw[_W_label], wp->entries[event->button]);
      }
      return wp->selected;
    case WIDGET_EVENT_KEY:
      {
        int i = xitk_browser_list_nav (event, (const char * const *)wp->entries, wp->num_entries, wp->selected, _COMBO_BROWSER_LINES);
        if (i >= 0) {
          if (i != wp->selected) {
            _combo_close (wp, 1);
            wp->selected = i;
            xitk_label_change_label (wp->iw[_W_label], wp->entries[i]);
            if (wp->callback)
              wp->callback (wp->iw[_W_combo], wp->w.userdata, i, event->modifier);
          }
          return 1;
        } 
      }
      break;
    case WIDGET_EVENT_WIN_POS:
      if (wp->xwin) {
        xitk_rect_t wr = {
          .x = (wp->iw[_W_label]->pos.w & 0xffff) + event->x,
          .y = ((wp->iw[_W_label]->pos.w + wp->iw[_W_label]->size.w) >> 16) + event->y,
          .width = XITK_INT_KEEP,
          .height = XITK_INT_KEEP
        };
        xitk_window_move_resize (wp->xwin, &wr);
        xitk_window_flags (wp->xwin, XITK_WINF_VISIBLE | XITK_WINF_ICONIFIED, XITK_WINF_VISIBLE);
        xitk_window_raise_window (wp->xwin);
        xitk_window_set_input_focus (wp->xwin);
        /* No widget focused, give focus to the first one */
        if (wp->widget_list->widget_focused == NULL)
          xitk_set_focus_to_next_widget (wp->widget_list, 0, 0);
        return 1;
      }
      break;
    default: ;
  }
  return 0;
}

/* ************************* END OF PRIVATES ******************************* */

/*
 *
 */
const char *xitk_combo_get_current_entry_selected(xitk_widget_t *w) {
  _combo_private_t *wp;

  xitk_container (wp, w, w);
  if (!wp)
    return NULL;

  if ((wp->w.type & WIDGET_TYPE_MASK) == WIDGET_TYPE_COMBO) {
    if (wp->entries && wp->selected >= 0)
      return (wp->entries[wp->selected]);
  }
  return NULL;
}

/*
 *
 */
void xitk_combo_update_list(xitk_widget_t *w, const char *const *const list, int len) {
  _combo_private_t *wp;

  xitk_container (wp, w, w);
  if (!wp)
    return;

  if ((wp->w.type & WIDGET_TYPE_MASK) == WIDGET_TYPE_COMBO) {
    free (wp->entries);
    wp->num_entries = len;
    wp->entries     = _combo_copy_string_list (list, &wp->num_entries);
    wp->selected    = -1;

    if (wp->xwin)
      xitk_browser_update_list (wp->iw[_W_browser],
        (const char* const*)wp->entries, NULL, wp->num_entries, 0);
  }
}

/*
 *
 */
static xitk_widget_t *_combo_create (const xitk_combo_widget_t *c,
  _combo_private_t *wp, int visible, int enable) {
  if (!wp->iw[_W_label] || !wp->iw[_W_button]) {
    xitk_widgets_delete (wp->iw + _W_label, 2);
    free (wp);
    return NULL;
  }

  _combo_hull (wp);

  wp->xwin = NULL;

  wp->iw[_W_combo]      = &wp->w;
  wp->parent_wkey       = c->parent_wkey;
  wp->callback          = c->callback;

  wp->w.state &= ~(XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE);
  wp->w.state |= visible ? XITK_WIDGET_STATE_VISIBLE : 0;
  wp->w.state |= enable ? XITK_WIDGET_STATE_ENABLE : 0;

  //  wp->w.x = wp->w.y = wp->w.width = wp->w.height = 0;
  wp->w.type         = WIDGET_GROUP | WIDGET_TYPE_COMBO | WIDGET_KEYABLE;
  wp->w.event        = notify_event;

  xitk_widget_set_focus_redirect (wp->iw[_W_label], wp->iw[_W_button]);
  xitk_widget_set_focus_redirect (&wp->w, wp->iw[_W_button]);

  return _xitk_new_widget_apply (&c->nw, &wp->w);
}

/*
 *
 */
xitk_widget_t *xitk_combo_create (const xitk_combo_widget_t *c, xitk_skin_config_t *skonfig) {
  _combo_private_t *wp;

  wp = (_combo_private_t *)xitk_widget_new (&c->nw, sizeof (*wp));
  if (!wp)
    return NULL;

  wp->num_entries = 0x7fffffff;
  wp->entries = _combo_copy_string_list (c->entries, &wp->num_entries);
  wp->selected = (XITK_0_TO_MAX_MINUS_1 (c->select, wp->num_entries)) ? c->select : -1;

  {
    xitk_button_widget_t b = {
      .nw = {
        .wl = wp->w.wl,
        .group = &wp->w,
        .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE,
        .mode_mask = WIDGET_GROUP_MEMBER | WIDGET_GROUP_COMBO,
        .mode_value = WIDGET_GROUP_MEMBER | WIDGET_GROUP_COMBO,
        .skin_element_name = c->nw.skin_element_name,
        .userdata = wp,
        .tips = c->nw.tips
      },
      .state_callback = _combo_rollunroll
    };
    xitk_label_widget_t lbl = {
      .nw = {
        .wl = wp->w.wl,
        .group = &wp->w,
        .add_state = XITK_WIDGET_STATE_ENABLE | XITK_WIDGET_STATE_VISIBLE,
        .mode_mask = WIDGET_TABABLE | WIDGET_GROUP_MEMBER | WIDGET_GROUP_COMBO,
        .mode_value = WIDGET_GROUP_MEMBER | WIDGET_GROUP_COMBO,
        .skin_element_name = c->nw.skin_element_name,
        .userdata = wp,
        .tips = c->nw.tips
      },
      .label = (wp->selected >= 0) ? wp->entries[wp->selected] : ""
    };

    /* Create label and button (skinable) */
    wp->iw[_W_label] = xitk_label_create (&lbl, skonfig);
    wp->iw[_W_button] = xitk_button_create (&b, skonfig);
  }

  {
    const xitk_skin_element_info_t *info = xitk_skin_get_info (skonfig, c->nw.skin_element_name);
    return _combo_create (c, wp,
      info ? (info->visibility ? 1 : -1) : 0, info ? info->enability : 0);
  }
}

/*
 *  ******************************************************************************
 */
xitk_widget_t *xitk_noskin_combo_create (const xitk_combo_widget_t *c, int x, int y, int width, int height) {
  _combo_private_t     *wp;

  wp = (_combo_private_t *)xitk_widget_new (&c->nw, sizeof (*wp));
  if (!wp)
    return NULL;

  wp->w.skin_element_name[0] = 0;

  wp->num_entries = 0x7fffffff;
  wp->entries = _combo_copy_string_list (c->entries, &wp->num_entries);
  wp->selected = (XITK_0_TO_MAX_MINUS_1 (c->select, wp->num_entries)) ? c->select : -1;

  /* Create label and button (skinable) */
  {
    char fontname[128];
    xitk_label_widget_t lbl = {
      .nw = {
        .wl = wp->w.wl,
        .group = &wp->w,
        .add_state = c->nw.add_state ? c->nw.add_state : XITK_WIDGET_STATE_KEEP,
        .mode_mask = WIDGET_TABABLE | WIDGET_GROUP_MEMBER | WIDGET_GROUP_COMBO,
        .mode_value = WIDGET_GROUP_MEMBER | WIDGET_GROUP_COMBO,
        .skin_element_name = "XITK_NOSKIN_INNER",
        .userdata = wp,
        .tips = c->nw.tips
      },
      .label = (wp->selected >= 0) ? wp->entries[wp->selected] : ""
    };
    xitk_button_widget_t b = {
      .nw = {
        .wl = wp->w.wl,
        .group = &wp->w,
        .add_state = c->nw.add_state ? c->nw.add_state : XITK_WIDGET_STATE_KEEP,
        .mode_mask = WIDGET_GROUP_MEMBER | WIDGET_GROUP_COMBO,
        .mode_value = WIDGET_GROUP_MEMBER | WIDGET_GROUP_COMBO,
        .userdata = wp,
        .tips = c->nw.tips
      },
      .symbol = XITK_SYMBOL_DOWN,
      .state_callback = _combo_rollunroll
    };

    XITK_DEFAULT_FONT_NAME (fontname, DEFAULT_FONT_FMT, height);

    wp->iw[_W_label] = xitk_noskin_label_create (&lbl, x, y, (width - height), height, fontname);
    wp->iw[_W_button] = xitk_noskin_button_create (&b, x + (width - height), y, height, height);
  }

  return _combo_create (c, wp, 0, 0);
}
