/**
 * \file pappsomspp/processing/cbor/psm/psmfilereaderbase.h
 * \date 05/07/2025
 * \author Olivier Langella
 * \brief Base class to read CBOR PSM file
 */

/*******************************************************************************
 * Copyright (c) 2025 Olivier Langella <Olivier.Langella@universite-paris-saclay.fr>.
 *
 * This file is part of PAPPSOms-tools.
 *
 *     PAPPSOms-tools is free software: you can redistribute it and/or modify
 *     it under the terms of the GNU General Public License as published by
 *     the Free Software Foundation, either version 3 of the License, or
 *     (at your option) any later version.
 *
 *     PAPPSOms-tools is distributed in the hope that it will be useful,
 *     but WITHOUT ANY WARRANTY; without even the implied warranty of
 *     MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 *     GNU General Public License for more details.
 *
 *     You should have received a copy of the GNU General Public License
 *     along with PAPPSOms-tools.  If not, see <http://www.gnu.org/licenses/>.
 *
 ******************************************************************************/
#pragma once

#include "pappsomspp/core/processing/cbor/cborstreamreaderinterface.h"
#include "pappsomspp/core/processing/uimonitor/uimonitorinterface.h"
#include "psmproteinmap.h"
#include "pappsomspp/core/peptide/peptide.h"
#include "pappsomspp/core/massspectrum/qualifiedmassspectrum.h"
#include <QCborArray>


namespace pappso
{
namespace cbor
{
namespace psm
{


struct PsmFile
{
  QString name;
};


/**
 * @todo write docs
 */
class PsmFileReaderBase : public CborStreamReaderInterface
{
  public:
  /**
   * Default constructor
   */
  PsmFileReaderBase();

  /**
   * Destructor
   */
  virtual ~PsmFileReaderBase();
  void readCbor(QFile *cborp, pappso::UiMonitorInterface &monitor);

  struct PsmProteinRef
  {
    QString accession;
    std::vector<std::size_t> positions;
  };

  virtual void close();

  protected:
  void readRoot(pappso::UiMonitorInterface &monitor);
  virtual void readInformations(pappso::UiMonitorInterface &monitor);
  virtual void readLog(pappso::UiMonitorInterface &monitor);
  virtual void readParameterMap(pappso::UiMonitorInterface &monitor);
  virtual void readProteinMap(pappso::UiMonitorInterface &monitor);
  virtual void readSample(pappso::UiMonitorInterface &monitor);
  virtual void readScan(pappso::UiMonitorInterface &monitor);
  virtual void readPsm(pappso::UiMonitorInterface &monitor);
  virtual void psmReady(pappso::UiMonitorInterface &monitor);
  virtual void logReady(pappso::UiMonitorInterface &monitor);
  virtual void informationsReady(pappso::UiMonitorInterface &monitor);
  virtual void parameterMapReady(pappso::UiMonitorInterface &monitor);
  virtual void fastaFilesReady(pappso::UiMonitorInterface &monitor);
  virtual void proteinMapReady(pappso::UiMonitorInterface &monitor);
  virtual void sampleListStarted(pappso::UiMonitorInterface &monitor);
  virtual void sampleStarted(pappso::UiMonitorInterface &monitor);
  virtual void scanStarted(pappso::UiMonitorInterface &monitor);
  virtual void scanFinished(pappso::UiMonitorInterface &monitor);
  virtual void sampleFinished(pappso::UiMonitorInterface &monitor);
  virtual void sampleListFinished(pappso::UiMonitorInterface &monitor);
  PsmFile readPsmFile(bool &is_ok);
  void writePsmFile(CborStreamWriter &writer, const PsmFile &psm_file);
  void writePsmFileList(CborStreamWriter &writer, const std::vector<PsmFile> &file_list);
  PsmProteinRef readPsmProteinRef(bool &is_ok);


  pappso::PeptideSp getCurrentPsmPeptideSp() const;
  pappso::QualifiedMassSpectrumSPtr getCurrentQualifiedMassSpectrumSPtr() const;


  bool currentProteinRefListContainsDecoy() const;
  bool currentProteinRefListContainsTarget() const;

  /** @brief convenient function do compute precusor ion mass
   */
  double getPrecursorMass(double mz_prec, uint charge) const;


  protected:
  QStringList m_targetFastaFiles;
  QStringList m_decoyFastaFiles;
  PsmProteinMap m_proteinMap;
  QString m_currentSampleName;
  PsmFile m_currentPeaklistFile;
  std::vector<PsmFile> m_currentIdentificationFileList;
  QCborMap m_cborInformations;
  QCborArray m_cborLog;
  QCborMap m_cborParameterMap;
  QCborMap m_cborScanId;
  QCborMap m_cborScanPrecursor;
  QCborMap m_cborScanMs2;
  QCborMap m_cborScanPsmEval;
  QCborMap m_cborScanPsmProps;
  QString m_currentPsmProforma;
  std::vector<PsmProteinRef> m_currentPsmProteinRefList;
};

} // namespace psm
} // namespace cbor
} // namespace pappso
