#
# Copyright 2024 Centreon (http://www.centreon.com/)
#
# Centreon is a full-fledged industry-strength solution that meets
# the needs in IT infrastructure and application monitoring for
# service performance.
#
# Licensed under the Apache License, Version 2.0 (the "License");
# you may not use this file except in compliance with the License.
# You may obtain a copy of the License at
#
#     http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS,
# WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
# See the License for the specific language governing permissions and
# limitations under the License.
#

package network::3com::snmp::mode::components::psu;

use strict;
use warnings;

my %map_status = (
    1 => 'active',
    2 => 'deactive',
    3 => 'not-install',
    4 => 'unsupport',
);

# In MIB 'a3com-huawei-splat-devm'
my $mapping = {
    hwDevMPowerStatus => { oid => '.1.3.6.1.4.1.43.45.1.2.23.1.9.1.2.1.2', map => \%map_status },
};
my $oid_hwdevMPowerStatusEntry = '.1.3.6.1.4.1.43.45.1.2.23.1.9.1.2.1';

sub load {
    my ($self) = @_;
    
    push @{$self->{request}}, { oid => $oid_hwdevMPowerStatusEntry };
}

sub check {
    my ($self) = @_;
    
    $self->{output}->output_add(long_msg => "Checking power supplies");
    $self->{components}->{psu} = {name => 'psus', total => 0, skip => 0};
    return if ($self->check_filter(section => 'psu'));
    
    foreach my $oid ($self->{snmp}->oid_lex_sort(keys %{$self->{results}->{$oid_hwdevMPowerStatusEntry}})) {
        next if ($oid !~ /^$mapping->{hwDevMPowerStatus}->{oid}\.(.*)$/);
        my $instance = $1;
        my $result = $self->{snmp}->map_instance(mapping => $mapping, results => $self->{results}->{$oid_hwdevMPowerStatusEntry}, instance => $instance);
        
        next if ($result->{hwDevMPowerStatus} =~ /not-install/i && 
                 $self->absent_problem(section => 'psu', instance => $instance));
        next if ($self->check_filter(section => 'psu', instance => $instance));
        $self->{components}->{psu}->{total}++;

        $self->{output}->output_add(long_msg => sprintf("Power supply '%s' status is '%s' [instance: %s]", 
                                    $instance, $result->{hwDevMPowerStatus}, 
                                    $instance));
        my $exit = $self->get_severity(label => 'default', section => 'psu', value => $result->{hwDevMPowerStatus});
        if (!$self->{output}->is_status(value => $exit, compare => 'ok', litteral => 1)) {
            $self->{output}->output_add(severity => $exit,
                                        short_msg => sprintf("Power supply '%s' status is %s", 
                                                             $instance, $result->{hwDevMPowerStatus}));
        }
    }
}

1;
