; SOFT (Second-Order Functions and Theorems) Library
;
; Copyright (C) 2020 Kestrel Institute (http://www.kestrel.edu)
;
; License: A 3-clause BSD license. See the LICENSE file distributed with ACL2.
;
; Author: Alessandro Coglio (coglio@kestrel.edu)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package "SOFT")

(include-book "core")

(include-book "kestrel/event-macros/cw-event" :dir :system)
(include-book "kestrel/event-macros/make-event-terse" :dir :system)
(include-book "kestrel/event-macros/restore-output" :dir :system)
(include-book "std/system/maybe-pseudo-event-formp" :dir :system)
(include-book "kestrel/utilities/er-soft-plus" :dir :system)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defxdoc+ defunvar-implementation
  :parents (soft-implementation defunvar)
  :short "Implementation of @(tsee defunvar)."
  :order-subtopics t
  :default-parent t)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define defunvar-fn ((inputs true-listp)
                     (call pseudo-event-formp "Call to @(tsee defunvar).")
                     (ctx "Context for errors.")
                     state)
  :returns (mv (erp "@(tsee booleanp) flag of the
                     <see topic='@(url acl2::error-triple)'>error
                     triple</see>.")
               (event maybe-pseudo-event-formp)
               state)
  :verify-guards nil
  :short "Validate the inputs to @(tsee defunvar)
          and generate the event form to submit."
  :long
  (xdoc::topstring-p
   "Similary to @(tsee *-listp),
    any @('*') and @('=>') symbol (i.e. in any package) is allowed.")
  (b* ((wrld (w state))
       ((unless (>= (len inputs) 4))
        (er-soft+ ctx t nil
                  "At least four inputs must be supplied, not ~n0."
                  (len inputs)))
       (funvar (first inputs))
       (arguments (second inputs))
       (arrow (third inputs))
       (result (fourth inputs))
       (options (nthcdr 4 inputs))
       ((unless (symbolp funvar))
        (er-soft+ ctx t nil
                  "The first input must be a symbol, but ~x0 is not."
                  funvar))
       ((unless (*-listp arguments))
        (er-soft+ ctx t nil
                  "The second input must be a list (* ... *), but ~x0 is not."
                  arguments))
       ((unless (and (symbolp arrow)
                     (equal (symbol-name arrow) "=>")))
        (er-soft+ ctx t nil
                  "The third input must be =>, but ~x0 is not."
                  arrow))
       ((unless (and (symbolp result)
                     (equal (symbol-name result) "*")))
        (er-soft+ ctx t nil
                  "The fourth input must be *, but ~x0 is not."
                  result))
       ((unless (or (null options)
                    (and (= (len options) 2)
                         (eq (car options) :print))))
        (er-soft+ ctx t nil
                  "After the * input there may be at most one :PRINT option, ~
                   but instead ~x0 was supplied."
                  options))
       (print (if options
                  (cadr options)
                nil))
       ((unless (member-eq print '(nil :all)))
        (er-soft+ ctx t nil
                  "The :PRINT input must be NIL or :ALL, but ~x0 is not."
                  print))
       ((when (funvarp funvar wrld))
        (b* ((arity (arity funvar wrld)))
          (if (= arity (len arguments))
              (prog2$ (cw "~%The call ~x0 is redundant.~%" call)
                      (value `(value-triple :invisible)))
            (er-soft+ ctx t nil "A function variable ~x0 with arity ~x1 ~
                                 already exists.~%" funvar arity))))
       (event `(progn
                 (defstub ,funvar ,arguments ,arrow ,result)
                 (table function-variables ',funvar nil)
                 (value-triple ',funvar)))
       (event (restore-output? (eq print :all) event)))
    (value event)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defsection defunvar-macro-definition
  :short "Definition of the @(tsee defunvar) macro."
  :long
  "@(def defunvar)
   @(def acl2::defunvar)"

  (defmacro defunvar (&whole call &rest inputs)
    `(make-event-terse (defunvar-fn
                         ',inputs
                         ',call
                         (cons 'defunvar ',(if (consp inputs) (car inputs) nil))
                         state)))

  (defmacro acl2::defunvar (&rest inputs)
    `(defunvar ,@inputs)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defsection show-defunvar
  :short "Show the event form generated by @(tsee defunvar),
          without submitting them."
  :long
  "@(def show-defunvar)
   @(def acl2::show-defunvar)"

  (defmacro show-defunvar (&whole call
                                  funvar arguments arrow result &key print)
    `(defunvar-fn
       ',funvar
       ',arguments
       ',arrow
       ',result
       ',print
       ',call
       (cons 'defunvar ',funvar)
       state))

  (defmacro acl2::show-defunvar (&rest args)
    `(show-defunvar ,@args)))
